% Code to reproduce figure 1 in Policy During an Epidemic With
% Super-Spreaders, Van Vlokhoven (2020)
% Relationship between fraction susceptible and effective reproduction
% number (dependent on the degree distribution)


clear all

%% parameters
lambda_R = 0.4975;  %recovery rate
lambda_D = 0.0025;  %death rate
gamma = lambda_R+lambda_D;
p = 0.07;   %transmission risk

d = [25 10];    %possible values of degree
frac = [0.3 0.7]; %mass of agents for each degree 
k = length(d);  %number of degree states
d_exp = d*frac';    %expected degree

epsilon = 0.001; %fraction infected initially 

TIME = 100; %number of time periods
dt = 1/2;       %time steps
T_grid = 1:dt:TIME; %time grid
nT = length(T_grid);

%% solve model with degree distribution
% create matrices with mass of agents in each state over time for each
% degree, and initialize firs period
s_d = NaN(nT,k);    %susceptible over time for each degree state
s_d(1,:) = (1-epsilon*d/d_exp); %reflects that those with a higher degree are more likely to be infected in first period
i_d = NaN(nT,k);   %infected population
i_d(1,:) = 1-s_d(1,:);
s_d_out = NaN(nT,k);    %mass that becomes infected in a period
r_d = NaN(nT,k);        %mass recovered
r_d(1,:) = 0;
f = NaN(nT,1);  %probability that a given social interaction is with an infectious agent
frac_alive = NaN(nT,k);
frac_alive(1,:) = 1;
% iterate model forward over time
for t = 1:nT-1
    frac_alive(t,:) = s_d(t,:) + i_d(t,:) + r_d(t,:);
    
    f(t) = ((i_d(t,:).*d)*frac')/((d.*frac_alive(t,:))*frac');     %need to update for people dying
  
    s_d_out(t+1,:) = dt*p*f(t)*d.*s_d(t,:);
  
    s_d(t+1,:) = s_d(t,:)-s_d_out(t+1,:);

    i_d(t+1,:) = (1-dt*gamma)*i_d(t,:)+s_d_out(t+1,:);
 
    r_d(t+1,:) = r_d(t,:)+dt*lambda_R*i_d(t,:);
    
%     R_alt(t+1,:) = R_alt(t,:) + dt*lambda_R*(frac-s_d(t,:)-R_alt(t,:)*(lambda_R+lambda_D)/lambda_R);
    
end
frac_alive(nT,:) = s_d(nT,:) + i_d(nT,:) + r_d(nT,:);
f(nT) = ((i_d(nT,:).*d)*frac')/((d.*frac_alive(nT,:))*frac');

%take weighted average over different degrees
s_d_all = s_d*frac';       
i_d_all = i_d*frac';
r_d_all = r_d*frac';

death_d = r_d_all * lambda_D/lambda_R;  %fraction that has died
death_d_ind = r_d * lambda_D/lambda_R; %fraction that has died by degree

L = (frac_alive*frac')';

%effective reproduction number (equation 9 in paper)
reproduction = (((repmat(d,nT,1).*s_d)*frac')/d_exp) .* (p/(lambda_R) * ((repmat(d,nT,1).*i_d)*frac')./i_d_all);    


%% solve model when degree is identical
d = 14.5;

%%%%%%%%%
%%% With the following line commented out, figure 1a is generated
%%% Uncomment this line to generate figure 1b, this ensures that the effective reproduction number at time 1 is 
%%% identical to Re with a degree distribution
% p = lambda_R*reproduction(1)/d(1);
%%%%%%%%%

s_1 = NaN(nT,1);
s_1(1,:) = (1-epsilon); %reflects that those with a higher degree are more likely to be infected

i_1 = NaN(nT,1);
i_1(1,:) = 1-s_1(1,:);

s_1_out = NaN(nT,1);

r_1 = NaN(nT,1);
r_1(1,:) = 0;
f1 = NaN(nT,1);
frac_alive1 = NaN(nT,1);
frac_alive1(1,:) = 1;
for t = 1:nT-1
    frac_alive1(t) = s_1(t) + i_1(t) + r_1(t);
    
    f1(t) = ((i_1(t).*d))/((d.*frac_alive1(t)));     %need to update for people dying
  
    s_1_out(t+1) = dt*p*f1(t)*d.*s_1(t);
  
    s_1(t+1) = s_1(t,:)-s_1_out(t+1);

    i_1(t+1) = (1-dt*gamma)*i_1(t)+s_1_out(t+1);
   
    r_1(t+1) = r_1(t,:)+dt*lambda_R*i_1(t);  
end
frac_alive1(nT) = s_1(nT) + i_1(nT) + r_1(nT);
f1(nT) = ((i_1(nT).*d))/((d.*frac_alive1(nT)));

s_1_all = s_1;
i_1_all = i_1;
r_1_all = r_1;

death_1 = r_1_all * lambda_D/lambda_R;
death_1_ind = r_1 * lambda_D/lambda_R;

L = frac_alive1';

reproduction1 = s_1 .* (p/(lambda_R) * (repmat(d,nT,1).*i_1)./i_1_all);


%% create plot
S_plot2 = 0.01:0.01:1;
horizontal = ones(length(S_plot2));


figure(1)
plot(s_d_all(1:nT-50),reproduction(1:nT-50),'linewidth',2,'Color','k');
hold on
plot(s_1_all(1:nT-50),reproduction1(1:nT-50),'--','linewidth',2,'Color','k');
plot(S_plot2,horizontal,':','linewidth',0.3,'Color','k');
scatter(s_d_all(1:2:nT-50),reproduction(1:2:nT-50),[],[0.2010 0.7450 0.9330],'filled')
scatter(s_1_all(1:2:nT-50),reproduction1(1:2:nT-50),[],[0.2010 0.7450 0.9330],'filled')
legend('Degree distribution','Same degree')
ylabel('Effective Reproduction Number','FontSize',24)
xlabel('S (average)','FontSize',24)
ax = gca;
ax.YAxis.Limits = ([0 3]);
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off


