% Code to reproduce figure 10 in Policy During an Epidemic With
% Super-Spreaders, Van Vlokhoven (2020)
% Evolution of epidemic when a planner has the option to set a tax on the
% consumption of the social good

clear all

par.T = 100;    %(1 period is a week), so solve for T weeks
par.dt = 1/2;   %time steps
par.T_grid = 1:par.dt:par.T;
par.nT = length(par.T_grid);

par.beta = 0.96^(par.dt/52);     %dicounting
  
par.phi = [0.07 0.2];   %weight utility function social good
par.nphi = length(par.phi);
par.g = [0.7 0.3];  %density phi
par.g = par.g/sum(par.g);
par.E_phi = par.phi*par.g'; %expectation phi

par.epsilon = 0.02;     %fraction infected initially

par.alpha = 0.7;         %returns to scale production function

%disease parameters
par.p = 0.07;       %transmission risk
par.lambda_R = 0.4975*par.dt;       %recovery rate
par.lambda_D = 0.0025*par.dt;       %death rate

%solve steady state (pre-disease onset) 
par.As = (par.E_phi/(1-par.E_phi))^(1-par.alpha);   %set A_s such that P_s is 1 in economy with no infections (Ar=L=1)
par.B_ss = ((par.As).^(par.alpha/(par.alpha-1))+par.As)./((1+(par.As).^(1/(par.alpha-1))).^par.alpha);  %Budget, Ps=1,L=1,Ar=1
par.cr_ss = (1-par.phi)*par.B_ss;   %consumption regular good
par.cs_ss = par.phi*par.B_ss;       %consumption social good
par.CS_ss = par.cs_ss*par.g';       %aggregate consumption social good

par.d_CS = 1;  %parameter that governs to what extent degree depends on aggregate consumption (in paper set to 1)

%%%%%%% value of life
par.value_life = 16*(52/par.dt)*par.B_ss;    %B_ss is income per capita per week (and 52 weeks in a year). 10 qaly (= 1million dollar) corresponds to 16 times GDP per capita

% set parameters to get following degree distribution in steady state
par.degree_target = [10 25]*par.dt;     %degree during steady state (before onset of epidemic)
par.a = (par.degree_target(2)-par.degree_target(1))/((par.CS_ss.^(par.d_CS))*par.B_ss*(par.phi(2)-par.phi(1)));
par.b = par.degree_target(1) - par.a*(par.CS_ss.^(par.d_CS))*par.cs_ss(1);
%degree = b + a * CS^d_CS * cs

%% Solve optimal policy
%number of of periods for which the policy cannot change
par.pol = [2 2 2 2 2 2 3 3 3 5 5 8 10 50 0.5]/par.dt; 
 
% initial guess on policy (i.e., tax on social consumption)
x0 = [0.01    0.02    0.03    0.04    0.0547    0.1465    0.2636    0.3669    0.3968    0.3698    0.3158    0.2577    0.1950    0.0731];

%lower bound and upper bound for tax social consumption (for fmincon)
lb = 0*ones(14,1);
ub = 0.7*ones(14,1);

options = optimoptions(@fmincon,'MaxIterations',1000,'MaxFunctionEvaluations',600,'OptimalityTolerance',1e-6,'StepTolerance',1e-8,'UseParallel',true,'Display','iter');
[liberal_policy,liberal_val,exitflag2,output2] = fmincon(@(x) -Liberal(x,par),x0,[],[],[],[],lb,ub,[],options)


%% given the optimal policy, calculate consumption and evolution of epidemic
% parameters
T = par.T;    %(1 period is a week), so solve for T weeks
nT = par.nT;
dt = par.dt;
beta = par.beta;     %dicounting
phi = par.phi; 
nphi = par.nphi;
g = par.g;  %density phi
epsilon = par.epsilon;     %fraction infected initially
alpha = par.alpha;
As = par.As;   %set A_s such that P_s is 1 in economy with no infections (Ar=L=1)

%disease parameters
p = par.p;
lambda_R = par.lambda_R;
lambda_D = par.lambda_D;

%steady state values
B_ss = par.B_ss;  %Ps=1,L=1,Ar=1
cr_ss = par.cr_ss;
cs_ss = par.cs_ss;
CS_ss = par.CS_ss;

d_CS = par.d_CS;

a = par.a;
b = par.b;

degree_ss = b+a*cs_ss*(CS_ss.^(d_CS));
degree_ss_exp = degree_ss*g';

value_life = par.value_life;    %B_ss is income per capita per week. 10 qaly (= 1 million dollar) corresponds to 16 times GDP per capita
u_lowerbar = (1-beta)*value_life/B_ss - log(B_ss);

u_ss = (1-phi).*log(cr_ss./(1-phi)) + phi.*log(cs_ss./phi) + u_lowerbar;
V_ss = u_ss/(1-beta); %value stady state when u_lowerbar = 0


%states capturing evolution of epidemic
S_phi = NaN(nT,nphi);
S_phi(1,:) = (1-epsilon*degree_ss/degree_ss_exp); %reflects that those with a higher degree are more likely to be infected
I_phi = NaN(nT,nphi);
I_phi(1,:) = epsilon*degree_ss/degree_ss_exp;
cs_eq = NaN(nT,nphi);   %consumption of social good over time for each type

eps1 = 1;
iter1 = 0;
gamma = 1;    %update rate time variables
gammaP = 0.2;  


x = liberal_policy;
x = [x 0];  %make sure in the last period the tax is zero
%construct the policy (for each period)
pol=par.pol;
tau = NaN(nT,1);
t2 = 0;
for i_p = 1:length(pol)
    t1 = t2+1;
    t2 = t1 + pol(i_p) - 1;
    tau(t1:t2) = x(i_p);    
end


% initial guess (from competitive)
f = [0.0252743187042286,0.0315361088717160,0.0384937589077554,0.0458856209819146,0.0534090551113552,0.0607311658411551,0.0675293822818887,0.0735622051250646,0.0786324875019579,0.0826044060288046,0.0854751533333284,0.0872706234549632,0.0880191731929458,0.0879085769457564,0.0869445209494787,0.0852170483177602,0.0829981245522543,0.0803002674826594,0.0772573395765691,0.0739547705282208,0.0705115108848695,0.0669984188091114,0.0634400834102358,0.0599122438679377,0.0564474824372110,0.0530775550217389,0.0498109970552529,0.0466837357732112,0.0436841243282816,0.0408660105360134,0.0382004992558925,0.0356710321427102,0.0333002703358012,0.0310701715387917,0.0289841225108286,0.0270424568152047,0.0252228201905419,0.0235285931395025,0.0219445214595160,0.0204739342894124,0.0191068191415239,0.0178358426051203,0.0166554334967115,0.0155504754773316,0.0145273679556938,0.0135745569816948,0.0126870142458037,0.0118610691644260,0.0110925541587225,0.0103776430299647,0.00971205543895595,0.00909000308337701,0.00850935055544727,0.00796991380799302,0.00746618183447577,0.00699481173980168,0.00655561513762544,0.00614576695458235,0.00576327549247168,0.00540539794425199,0.00507157660015885,0.00475811090801434,0.00446539849465172,0.00419133601957283,0.00393485767359254,0.00369476448084552,0.00346958442724948,0.00325961319904580,0.00306194723646885,0.00287649323340814,0.00270328286347493,0.00254034154805589,0.00238739402922272,0.00224413680466066,0.00210946142314619,0.00198306330450965,0.00186475668375714,0.00175356768132334,0.00164920056258254,0.00155075891846442,0.00145870662671513,0.00137213893978339,0.00129069727903355,0.00121420448951968,0.00114232678241126,0.00107455099118794,0.00101117599094138,0.000951463865230631,0.000895430332510578,0.000842609696589436,0.000792894216372060,0.000746169460045413,0.000702140070478877,0.000660715668886651,0.000621763597237614,0.000585229988415615,0.000550885683658638,0.000518607163978110,0.000488186478053196,0.000459528254417117,0.000432571963139843,0.000407240794393070,0.000383372265059624,0.000360884209750104,0.000339698882150445,0.000319748628173053,0.000300990387170909,0.000283365464502209,0.000266762046181693,0.000251122008616489,0.000236390484773483,0.000222515588547130,0.000209433281958183,0.000197178236859977,0.000185647623510999,0.000174828630753972,0.000164635342332353,0.000155031963586335,0.000145984698063585,0.000137461619666073,0.000129432620109271,0.000121869243807531,0.000114744675524297,0.000108033573701649,0.000101712053900457,9.57575610328782e-05,9.01488433547304e-05,8.48658639233226e-05,7.98897233638539e-05,7.52025489307624e-05,7.07875834639156e-05,6.66291459350626e-05,6.27140038634231e-05,5.90339683906835e-05,5.55733630349189e-05,5.23126535429221e-05,4.92400925624675e-05,4.63446648963853e-05,4.36160280070425e-05,4.10444624203175e-05,3.86208290039390e-05,3.63365312464825e-05,3.41834813377908e-05,3.21540692535698e-05,3.02411342927777e-05,2.84379386720775e-05,2.67381428842302e-05,2.51357825978315e-05,2.36252469261310e-05,2.22012579296538e-05,2.08588512450554e-05,1.95933577536085e-05,1.84003862186299e-05,1.72758068332094e-05,1.62157356286773e-05,1.52165197010547e-05,1.42747232177975e-05,1.33871141709190e-05,1.25506518453810e-05,1.17624749737459e-05,1.10198905496741e-05,1.03203632740756e-05,9.66150560868603e-06,9.04106841260570e-06,8.45693213797134e-06,7.90709856145241e-06,7.38968302870828e-06,6.90290718932425e-06,6.44509220007986e-06,6.01465237470508e-06,5.61008925856048e-06,5.22998610694913e-06,4.87300274604026e-06,4.53787079567045e-06,4.22338923359522e-06,3.92842028110259e-06,3.65188559027702e-06,3.39276271362342e-06,3.15008183723061e-06,2.92292275917411e-06,2.71041209542963e-06,2.51172069619112e-06,2.32606125615627e-06,2.15268610305597e-06,1.99088514945569e-06,1.83998399364046e-06,1.69934215620508e-06,1.56835143979880e-06,1.44643440031252e-06,1.33304291863920e-06,1.22765686297624e-06,1.12978283246652e-06,1.03895297378489e-06,9.54723863063914e-07,8.76675446310847e-07,8.04410032192724e-07,7.37551331754131e-07,6.75743540279571e-07,6.18650457116834e-07]';
L = [1,0.999975000000000,0.999944320772262,0.999907319764828,0.999863614662152,0.999813012628571,0.999755672054822,0.999691970476970,0.999622542897290,0.999548172797966,0.999469788309198,0.999388333387725,0.999304748097830,0.999219949816115,0.999134784628002,0.999050018775180,0.998966274588099,0.998884166340644,0.998804143255634,0.998726577941101,0.998651740152581,0.998579809159931,0.998510925792293,0.998445171169054,0.998382564121104,0.998323102431612,0.998266734093738,0.998213398862338,0.998163012846138,0.998115472303263,0.998070658836648,0.998028449849198,0.997988733676359,0.997951382798885,0.997916280342182,0.997883305199400,0.997852331269552,0.997823251804770,0.997795956583786,0.997770345209209,0.997746312566901,0.997723762260287,0.997702603061192,0.997682747062652,0.997664121804788,0.997646645983365,0.997630248418390,0.997614862478549,0.997600424827122,0.997586875340591,0.997574156834944,0.997562216284204,0.997551007495894,0.997540485946054,0.997530606820692,0.997521329984927,0.997512618602131,0.997504436281250,0.997496749896361,0.997489528273603,0.997482743294260,0.997476367033033,0.997470375308506,0.997464744063110,0.997459451095355,0.997454475279848,0.997449797959132,0.997445400621917,0.997441265005977,0.997437376208973,0.997433719694510,0.997430280293692,0.997427045396436,0.997424002806369,0.997421140829904,0.997418448750436,0.997415916287732,0.997413533601787,0.997411291713494,0.997409182049551,0.997407197324157,0.997405329664568,0.997403572005192,0.997401918109282,0.997400361680734,0.997398896746057,0.997397518212407,0.997396220553773,0.997394999234740,0.997393849482306,0.997392767273671,0.997391748714946,0.997390789907978,0.997389887470673,0.997389038125550,0.997388238592399,0.997387485935703,0.997386777393249,0.997386110248278,0.997385482140219,0.997384890834719,0.997384334167824,0.997383810005031,0.997383316488378,0.997382851860066,0.997382414458079,0.997382002711782,0.997381615099538,0.997381250129875,0.997380906501878,0.997380582986049,0.997380278420931,0.997379991709852,0.997379721816365,0.997379467722651,0.997379228507162,0.997379003218733,0.997378791054463,0.997378591256846,0.997378403111354,0.997378225944158,0.997378059119879,0.997377902039539,0.997377754138499,0.997377614884606,0.997377483776354,0.997377360341191,0.997377244133866,0.997377134734880,0.997377031749033,0.997376934804106,0.997376843549436,0.997376757654534,0.997376676807699,0.997376600706305,0.997376529060046,0.997376461612990,0.997376398123906,0.997376338365427,0.997376282123270,0.997376229195491,0.997376179391784,0.997376132532821,0.997376088449618,0.997376046982941,0.997376007982741,0.997375971307623,0.997375936824337,0.997375904407303,0.997375873938160,0.997375845305342,0.997375818403673,0.997375793133991,0.997375769402791,0.997375747121890,0.997375726208109,0.997375706582977,0.997375688172452,0.997375670906657,0.997375654719636,0.997375639549122,0.997375625336318,0.997375612025700,0.997375599564816,0.997375587904121,0.997375576996797,0.997375566798607,0.997375557267743,0.997375548364688,0.997375540052093,0.997375532294652,0.997375525058993,0.997375518313569,0.997375512028562,0.997375506175791,0.997375500728622,0.997375495661890,0.997375490951819,0.997375486575953,0.997375482513086,0.997375478743199,0.997375475247399,0.997375472007864,0.997375469007785,0.997375466231319,0.997375463663538,0.997375461290383,0.997375459098623,0.997375457075811,0.997375455210243,0.997375453490923,0.997375451907527,0.997375450450366,0.997375449110354,0.997375447878979,0.997375446748268,0.997375445710762,0.997375444759489,0.997375443887934]';
Ps = [0.927120075509146,0.914295372191969,0.900755699311531,0.888501671284327,0.876593804558082,0.866225995049189,0.856970409043824,0.849497960912450,0.843200448011832,0.838727548279146,0.835653425503534,0.833679724346914,0.832870113278464,0.832965471427779,0.834435728971348,0.835551448145170,0.838414521011413,0.841370907278822,0.845004357779187,0.849246414294413,0.853463554864283,0.857924250433844,0.862798615734999,0.867491328100412,0.872618980909487,0.877444199500986,0.882344135286599,0.887354441108444,0.892440965172141,0.897519624293433,0.902039958077445,0.906767826774937,0.911119517198258,0.915484312621682,0.920073320746903,0.923890530140902,0.927811274094640,0.931395692127294,0.935176380471816,0.938701769002761,0.942083122986862,0.945412835741664,0.947849662222716,0.951023565058839,0.953787540051018,0.956187958336386,0.958664969524082,0.961023938111015,0.963346554496083,0.965441031399850,0.967148509674378,0.968863379954282,0.970987969500087,0.972616064590501,0.973851664212448,0.975495616349396,0.976968220484238,0.978393086452068,0.979564783038367,0.980912645736359,0.981712909805110,0.982877479325009,0.983834784816814,0.984827717647646,0.985676966377497,0.986404835362600,0.987589373865170,0.987971425398085,0.988545453235288,0.989528594755279,0.989906104842987,0.990446363110315,0.991093838953819,0.991467969310683,0.991979500496486,0.992642086372318,0.992997521169611,0.993481855601402,0.993512961764826,0.994197620507823,0.994525569297555,0.994676293657113,0.994989059412448,0.995344558035112,0.995350344841757,0.996048683639804,0.996053988137502,0.996489952632512,0.996491081444096,0.996561668681710,0.996846266773636,0.996848340854391,0.996935602852332,0.997296817870909,0.997573491775964,0.997717283964915,0.997988341766137,0.997989236516748,0.997990133892184,0.998079384516811,0.998342796803836,0.998343444653965,0.998344057224336,0.998345531181048,0.998348546880219,0.998449367839441,0.998698186947822,0.998699428784955,0.998700593227249,0.998702690129341,0.998705687721111,0.998727263344119,0.999095286487123,0.999117923626692,0.999482062295431,0.999482065207351,0.999482148765149,0.999482401887546,0.999482404312016,0.999482870496529,0.999482872637773,0.999483524116402,0.999483776102764,0.999484437030338,0.999484795854115,0.999485713205941,0.999487496800640,0.999489993152043,0.999491020100772,0.999494935106302,0.999508076128903,0.999541701184656,0.999713892892540,0.999834843954959,0.999834844812869,0.999834845615409,0.999834846365354,0.999834847065448,0.999834847718375,0.999834848326738,0.999834848893049,0.999834849419723,0.999834849909070,0.999834850363299,0.999834850784514,0.999834851174717,0.999834851535812,0.999834851869607,0.999834852177817,0.999834852462070,0.999834852723907,0.999834852964787,0.999834853186093,0.999834853389132,0.999834853575141,0.999834853745289,0.999834853900678,0.999834854042351,0.999834854171292,0.999834854288427,0.999834854394628,0.999834854490717,0.999834854577468,0.999834854655606,0.999834854725812,0.999834854788725,0.999834854844945,0.999834854895030,0.999834854939505,0.999834854978856,0.999834855013539,0.999834855043978,0.999834855070565,0.999834855093666,0.999834855113619,0.999834855130736,0.999834855145307,0.999834855157596,0.999834855167847,0.999834855176287,0.999834855183120,0.999834855188533,0.999834855192699,0.999834855195772,0.999834855197894,0.999834855199194,0.999834855199785,0.999834855199773,0.999834855199251,0.999834855198301,0.999834855196999,0.999834855195409,0.999834855193591,0.999834855191595,0.999834855189467,0.999834855187245,0.999834855184963,0.999834855182651,0.999834855180334]';
CS = [0.0962214809530633,0.0934014537695784,0.0904610227576351,0.0878331990400656,0.0853107816119981,0.0831401347587302,0.0812228015486440,0.0796890161180378,0.0784061094129382,0.0774998022412319,0.0768783696317643,0.0764786833671662,0.0763118255276364,0.0763253123710393,0.0766138271441406,0.0768311025662758,0.0774005775043617,0.0779909574776402,0.0787210919523364,0.0795787340767488,0.0804356844924587,0.0813471807629878,0.0823491743614962,0.0833189594795471,0.0843849776710102,0.0853934124788144,0.0864230069162014,0.0874815110745771,0.0885619219595079,0.0896463779191459,0.0906160163452994,0.0916350831497316,0.0925771061661185,0.0935260507525511,0.0945282221772048,0.0953648173847728,0.0962273450454458,0.0970185005939818,0.0978559565188777,0.0986393671944977,0.0993930730257630,0.100137495973170,0.100683303591600,0.101396446616996,0.102018931811487,0.102560607276663,0.103120809177562,0.103655385644806,0.104182775592953,0.104659170731321,0.105048009600890,0.105439124874933,0.105924684128407,0.106297178876464,0.106580048853704,0.106957114684172,0.107295256087162,0.107622818531840,0.107892376415627,0.108202882114102,0.108387200330689,0.108655888816236,0.108876878031024,0.109106303460818,0.109302629102826,0.109470959278243,0.109745358029906,0.109833691400920,0.109966646170283,0.110194735474344,0.110282195901021,0.110407541422602,0.110557895904136,0.110644717782471,0.110763563742808,0.110917648095614,0.111000251608113,0.111112928332976,0.111120005283765,0.111279453964247,0.111355782186068,0.111390794965264,0.111463624906127,0.111546452149154,0.111547682055978,0.111710587954152,0.111711719598839,0.111813432354538,0.111813601334387,0.111830000405946,0.111896400960736,0.111896806775857,0.111917120968132,0.112001461982599,0.112066059615716,0.112099611595561,0.112162926970159,0.112163081565552,0.112163239799724,0.112184060925437,0.112245617290218,0.112245725788109,0.112245828815296,0.112246135588557,0.112246805407988,0.112270353146240,0.112328525505862,0.112328786454212,0.112329030846979,0.112329495453071,0.112330172374321,0.112335198011208,0.112421283402090,0.112426560551649,0.112511764977231,0.112511747370177,0.112511749726706,0.112511792866609,0.112511778232766,0.112511873133886,0.112511860204301,0.112512000162383,0.112512047276072,0.112512190954382,0.112512264465567,0.112512469505258,0.112512877930416,0.112513453798286,0.112513686016565,0.112514595030672,0.112517664553207,0.112525529678389,0.112565833170066,0.112594142796025,0.112594137430918,0.112594132397494,0.112594127676129,0.112594123248343,0.112594119096733,0.112594115204903,0.112594111557409,0.112594108139705,0.112594104938084,0.112594101939637,0.112594099132200,0.112594096504314,0.112594094045185,0.112594091744643,0.112594089593107,0.112594087581551,0.112594085701474,0.112594083944865,0.112594082304179,0.112594080772308,0.112594079342558,0.112594078008623,0.112594076764563,0.112594075604787,0.112594074524025,0.112594073517320,0.112594072580000,0.112594071707671,0.112594070896193,0.112594070141673,0.112594069440446,0.112594068789064,0.112594068184285,0.112594067623057,0.112594067102514,0.112594066619961,0.112594066172863,0.112594065758842,0.112594065375663,0.112594065021228,0.112594064693569,0.112594064390837,0.112594064111301,0.112594063853337,0.112594063615424,0.112594063396138,0.112594063194144,0.112594063008195,0.112594062837124,0.112594062679841,0.112594062535327,0.112594062402631,0.112594062280867,0.112594062169207,0.112594062066880,0.112594061973169,0.112594061887405,0.112594061808967,0.112594061737278,0.112594061671801,0.112594061612039,0.112594061557530,0.112594061507847,0.112594061462594,0.112594061421403]';

Ps = Ps./(1+tau);
CS = CS./(1+tau);

% construct the grid
dS = 0.025;
S_grid = 0:dS:1;     

dI = 0.03;   
I_grid = 0:dI:0.3; 

T_grid = 1:dt:T;

nS = length(S_grid);    %make sure nS  = nR
nI = length(I_grid);
nT = length(T_grid);

xs_grid = 0.2:(0.8/350):1.05;   %share of phi spent on social good

[I_mat,S_mat,XS_mat] = meshgrid(I_grid,S_grid,xs_grid);
I_tmp = I_mat(:,:,1);
S_tmp = S_mat(:,:,1);

alive = min(1-(1-S_mat - I_mat)*lambda_D/(lambda_D+lambda_R),1); 
Itilde_mat = I_mat./alive;

L1 = L;
Ps1 = Ps;
f1 = f;
CS1 = CS;

epsL = 1;
epsf = 1;
epsP = 1;
epsCS = 1;
a0 = a;  
while (epsL > 1e-3 || epsf > 1e-4 || epsP > 1e-3 || epsCS > 1e-3) && iter1 < 100      
    iter1 = iter1 + 1;

    L = gamma*L1 + (1-gamma)*L;
    Ps = gammaP*Ps1 + (1-gammaP)*Ps;
    f = gamma*f1 + (1-gamma)*f;
    CS = gammaP*CS1 + (1-gammaP)*CS;
    a = a0*CS.^d_CS;
    
    %income per capita per period (tax is rebated lump-sum)
    B = (((As*Ps).^(alpha/(alpha-1))+As*Ps.*(1+tau))./((1+(As*Ps).^(1/(alpha-1))).^alpha)).*(L.^(alpha-1));   

    V = NaN(nS,nI,nT,nphi);
    cs_fin = NaN(nS,nI,nT,nphi);
    cs_fin(:,:,nT,1)= repmat(phi(1)*B(end)/((1+tau(end))*Ps(end)),nS,nI,1);
    cs_fin(:,:,nT,2)= repmat(phi(2)*B(end)/((1+tau(end))*Ps(end)),nS,nI,1);
    % assume that after last period everyone that is I moves to R
    % in last period, consume steady state
    V(:,:,nT,1) = ( (1-phi(1))*log((B(end)-((1+tau(end))*Ps(end))*cs_fin(:,:,nT,1))/(1-phi(1))) + phi(1)*log(cs_fin(:,:,nT,1)/phi(1)) + u_lowerbar)/(1-beta); 
    V(:,:,nT,2) = ( (1-phi(2))*log((B(end)-((1+tau(end))*Ps(end))*cs_fin(:,:,nT,2))/(1-phi(2))) + phi(2)*log(cs_fin(:,:,nT,2)/phi(2)) + u_lowerbar)/(1-beta); 
 
    for i_phi = 1:nphi   
        for t = nT-1:-1:1       
            cs = XS_mat*phi(i_phi)*B(t)/((1+tau(t))*Ps(t)); %consumption social good
            cr = (1-XS_mat*phi(i_phi))*B(t);
            degree = b+a(t)*cs;
            S_out = p*f(t)*degree.*S_mat;
            Sprime = max(S_mat-S_out,S_grid(1));
            Iprime = I_mat*(1-lambda_R-lambda_D)+S_out;

            U = phi(i_phi)*log(cs) + (1-phi(i_phi))*log(cr) - phi(i_phi)*log(phi(i_phi)) - (1-phi(i_phi))*log(1-phi(i_phi)) + u_lowerbar;
            Wprime = interp2(I_tmp,S_tmp,V(:,:,t+1,i_phi),Iprime,Sprime,'linear',-inf);

            %optimize over consumption
            Wmatrix = U + beta*(1-lambda_D*Itilde_mat).*Wprime;     %
            [Wtemp,cs_ind] = max(Wmatrix,[],3);
            V(:,:,t,i_phi) = Wtemp;
            cs_fin(:,:,t,i_phi) = xs_grid(cs_ind)*phi(i_phi)*B(t)/((1+tau(t))*Ps(t));
        end

         %solve for time path S and I
        for t=1:nT-1
            cs_eq(t,i_phi) = interp2(I_tmp,S_tmp,cs_fin(:,:,t,i_phi),I_phi(t,i_phi),S_phi(t,i_phi),'linear');

            S_out = p*f(t)*(b+a(t)*cs_eq(t,i_phi)).*S_phi(t,i_phi);
            S_phi(t+1,i_phi) = S_phi(t,i_phi) - S_out; 
            I_phi(t+1,i_phi) = I_phi(t,i_phi)*(1-lambda_R-lambda_D)+S_out;
        end
        cs_eq(nT,i_phi) = interp2(I_tmp,S_tmp,cs_fin(:,:,nT,i_phi),I_phi(nT,i_phi),S_phi(nT,i_phi),'linear');


    end
    degree_eq = b+repmat(a,1,nphi).*cs_eq;

    frac_alive = 1-(1-S_phi-I_phi)*lambda_D/(lambda_R+lambda_D);

    L1 = frac_alive*g';         %mass of agents alive (and labor supply)
    f1 = ((degree_eq.*I_phi)*g')./((degree_eq.*frac_alive)*g'); %probability that a person you meet is infected 

    cs_demand = (cs_eq.*frac_alive)*g';
    Ps1 = ((max((As.^(1/alpha)) .* L1./(cs_demand.^(1/alpha)) - 1,0.00001)).^(alpha-1))./As;    %Price at which social market clears. make sure that term to the power is positive

    CS1 = cs_demand;

    eps1 = norm(L1-L) + norm(f1-f) + norm(Ps1-Ps);
    epsL = norm(L1-L);
    epsf = norm(f1-f);
    epsP = norm(Ps1-Ps);
    epsCS = norm(CS1-CS);
    fprintf('iter big loop=%d epsL=%1.8f epsf=%1.8f epsP=%1.8f epsCS=%1.8f\n',iter1,epsL,epsf,epsP,epsCS) 
end

B_fin = B;

I_overall = I_phi*g';

value1 = interp2(I_tmp,S_tmp,V(:,:,1,1),I_phi(1,1),S_phi(1,1),'linear');
value2 = interp2(I_tmp,S_tmp,V(:,:,1,2),I_phi(1,2),S_phi(1,2),'linear');
        
value_liberal = g(1) * value1 +  g(2) * value2;


%% create plots
nT2 = nT-100;

figure(1)
yyaxis left
plot(T_grid(1:nT2),I_overall(1:nT2),'linewidth',2,'Color','b')
hold on
plot(T_grid(1:nT2),I_phi((1:nT2),2),'linewidth',2,'Color','r')
plot(T_grid(1:nT2),I_phi((1:nT2),1),'linewidth',2,'Color','g')
plot(T_grid(1:nT2),f(1:nT2),'linewidth',2,'Color','m')
ylabel('Fraction of Initial Population','FontSize',24)
yyaxis right
plot(T_grid(1:nT2),1-frac_alive((1:nT2),:)*g','linewidth',2,'Color','k')
ylabel('Deaths as Fraction of Initial Population','FontSize',24)
legend('Infected','Infected high degree','Infected low degree','Prob meeting is infected','Death (right axis)')
% legend('Infected','Death (right axis)')
xlabel('Time (weeks)','FontSize',24)
ax = gca;
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
ax.YAxis(1).Limits = ([0, 0.3]);
ax.YAxis(2).Limits = ([0, 0.005]);
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off


figure(2)
yyaxis left
plot(T_grid(1:nT2),cs_eq((1:nT2),1),'-','linewidth',2,'Color',[0 0.4470 0.7410])
hold on
plot(T_grid(1:nT2),repmat(cs_ss(1),nT2,1),'--','linewidth',2,'Color',[0 0.4470 0.7410],'HandleVisibility','off')
plot(T_grid(1:nT2),cs_eq((1:nT2),2),'-','linewidth',2,'Color',[0.8500 0.3250 0.0980])
plot(T_grid(1:nT2),repmat(cs_ss(2),nT2,1),'--','linewidth',2,'Color',[0.8500 0.3250 0.0980],'HandleVisibility','off')
ylabel('Cons social good','FontSize',24)
yyaxis right
plot(T_grid(1:nT2),tau(1:nT2),'-','linewidth',2,'Color','k')
ylabel('Tax','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
legend('Consumption (low degree)','Consumption (high degree)','Tax (right axis)')
ax = gca;
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
ax.YAxis(1).Limits = ([0, 0.29]);
ax.YAxis(2).Limits = ([0, 0.57]);
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off


figure(3)
plot(T_grid(1:nT2),degree_eq((1:nT2),1)/degree_ss(1),'linewidth',2)
hold on
plot(T_grid(1:nT2),degree_eq((1:nT2),2)/degree_ss(2),'linewidth',2)
legend('Low degree','High degree')
ylabel('Degree relative to steady state','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
ax = gca;
ax.YAxis.Limits = ([0.4, 1.1]);
hold off


figure(4)
plot(T_grid(1:nT2),B_fin(1:nT2)/B_ss,'linewidth',2,'Color','k')
hold on
ylabel('Income per capita relative to steady state','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off






