% Code to reproduce figures 2 and 3 in Policy During an Epidemic With
% Super-Spreaders, Van Vlokhoven (2020)
% Effect of lockdown ("authoritarian") and discouragement ("liberal") policy in SIR model with degree
% distribution

clear all

%% parameters
lambda_R = 0.4975;  %recovery rate
lambda_D = 0.0025;  %death rate
gamma = lambda_R+lambda_D;
p = 0.07;   %transmission risk

d = [25 10];    %possible values of degree
frac = [0.3 0.7]; %mass of agents for each degree 
frac = frac/sum(frac);
k = length(d);  %number of degree states
d_exp = d*frac';    %expected degree

TIME = 50; %number of time periods
dt = 1/2;       %time steps
T_grid = 1:dt:TIME; %time grid
nT = length(T_grid);

epsilon = 0.001; %fraction infected initially

%%%%%%%%%%%%
%%% policy during the lockdown time period (in period 14 till 40
%%% limit social interactions to 6 for everyone (such that high degree agents 
%%% are more affected))
policy_strict = max(d)*ones(nT,1); 
policy_strict(14:40) = 6*ones(27,1);    
%%%%%%%%%%%%

%%%%%%%%%%%%
%%% policy during the encouragement time period (in period 14 till 40
%%% limit social interactions to 0.36 of original)
policy_liberal = ones(nT,1);
policy_liberal(14:40) = 0.36*policy_liberal(14:40); 
%%%%%%%%%%%%

%% Solve model
% create matrices with mass of agents in each state over time for each
% degree, and initialize firs period
% discouragement policy (liberal)
s_liberal = NaN(nT,k);      %susceptible
s_liberal(1,:) = (1-epsilon*d/d_exp);
i_liberal = NaN(nT,k);      %infected
i_liberal(1,:) = 1-s_liberal(1,:);
s_liberal_out = NaN(nT,k);  %become infected
r_liberal = NaN(nT,k);      %recovered
r_liberal(1,:) = 0;
frac_alive = NaN(nT-1,k);
f = NaN(nT-1,1);    %probability that a given social interaction is with an infectious agent

% initial conditions are the same for the lockdown policy (strict)
s_strict = s_liberal;
i_strict = i_liberal;
s_strict_out = s_liberal_out;
r_strict = r_liberal;
frac_alive2 = NaN(nT-1,k);
f2 = NaN(nT-1,1);


% iterate model forward over time
for t = 1:nT-1
    %if liberal
    frac_alive(t,:) = s_liberal(t,:) + i_liberal(t,:) + r_liberal(t,:);
   
    f(t) = ((i_liberal(t,:).*d)*frac')/((d.*frac_alive(t,:))*frac');     %need to update for people dying
  
    s_liberal_out(t+1,:) =  dt*p*f(t)*policy_liberal(t)*d.*s_liberal(t,:);
    
    s_liberal(t+1,:) = s_liberal(t,:)-s_liberal_out(t+1,:);   
    i_liberal(t+1,:) = (1-dt*gamma)*i_liberal(t,:)+s_liberal_out(t+1,:);   
    r_liberal(t+1,:) = r_liberal(t,:)+dt*lambda_R*i_liberal(t,:);

    %if strict
    frac_alive2(t,:) = s_strict(t,:) + i_strict(t,:) + r_strict(t,:);
   
    d_tmp = min(d,policy_strict(t));
    f2(t) = ((i_strict(t,:).*d_tmp)*frac')/((d_tmp.*frac_alive2(t,:))*frac');     %need to update for people dying
  
    s_strict_out(t+1,:) =  dt*p*f2(t)*d_tmp.*s_strict(t,:);
    
    s_strict(t+1,:) = s_strict(t,:)-s_strict_out(t+1,:);   
    i_strict(t+1,:) = (1-dt*gamma)*i_strict(t,:)+s_strict_out(t+1,:);   
    r_strict(t+1,:) = r_strict(t,:)+dt*lambda_R*i_strict(t,:);
    
end
death_liberal = (1-frac_alive)*frac';
death_strict = (1-frac_alive2)*frac';

s_strict_all = s_strict*frac';
s_liberal_all = s_liberal*frac';

i_strict_all = i_strict*frac';
i_liberal_all = i_liberal*frac';

%% create plots
% create shaded area in graph when policy is active
policy = zeros(nT,1); 
policy(15:40) = 0.004*ones(26,1); 

%%% figure 2 in paper
figure(1)
yyaxis left
plot(T_grid,i_liberal_all,'linewidth',2,'Color',[0.9290 0.6940 0.1250])
hold on
plot(T_grid,i_strict_all,'linewidth',2,'Color',[0.9500 0.2250 0.0580])
ylabel('Infected as Fraction of Initial Population','FontSize',24)
yyaxis right
plot(T_grid(1:end-1),death_liberal,'linewidth',2,'Color',[0 0.4470 0.7410])
plot(T_grid(1:end-1),death_strict,'linewidth',2,'Color','k')
h = fill(T_grid,policy,[0.3010 0.7450 0.9330]);
set(h,'facealpha',.2)
set(h,'edgecolor','none')
ylabel('Deaths as Fraction of Initial Population','FontSize',24)
legend('Infected Discouragement','Infected Lockdown','Death Discouragement','Death Lockdown')
xlabel('Time (weeks)','FontSize',24)
ax = gca;
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
set(gca, 'FontSize', 13)
hold off

%%% figure 3a in paper
policy(15:40) = 1*ones(26,1); 
% liberal
figure(2)
yyaxis left
plot(T_grid,i_liberal(:,1),'--','linewidth',2,'Color',[0.9290 0.6940 0.1250])
hold on
plot(T_grid,i_liberal(:,2),'-','linewidth',2,'Color',[0.9290 0.6940 0.1250])
ylabel('Infected as Fraction of Initial Population','FontSize',24)
yyaxis right
plot(T_grid,r_liberal(:,1),'--','linewidth',2,'Color',[0.4660 0.6740 0.1880])
plot(T_grid,r_liberal(:,2),'-','linewidth',2,'Color',[0.4660 0.6740 0.1880])
h = fill(T_grid,policy,[0.3010 0.7450 0.9330]);
set(h,'facealpha',.2)
set(h,'edgecolor','none')
ylabel('Immune as Fraction of Initial Population','FontSize',24)
legend('Infected high degree','Infected low degree','Immune high degree','Immune low degree')
xlabel('Time (weeks)','FontSize',24)
ax = gca;
ax.YAxis(1).Limits = ([0, 0.15]);
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
set(gca, 'FontSize', 13)
hold off

%%% figure 3b in paper
% authoritarian
figure(3)
yyaxis left
plot(T_grid,i_strict(:,1),'--','linewidth',2,'Color',[0.9290 0.6940 0.1250])
hold on
plot(T_grid,i_strict(:,2),'-','linewidth',2,'Color',[0.9290 0.6940 0.1250])
ylabel('Infected as Fraction of Initial Population','FontSize',24)
yyaxis right
plot(T_grid,r_strict(:,1),'--','linewidth',2,'Color',[0.4660 0.6740 0.1880])
plot(T_grid,r_strict(:,2),'-','linewidth',2,'Color',[0.4660 0.6740 0.1880])
h = fill(T_grid,policy,[0.3010 0.7450 0.9330]);
set(h,'facealpha',.2)
set(h,'edgecolor','none')
ylabel('Immune as Fraction of Initial Population','FontSize',24)
legend('Infected high degree','Infected low degree','Immune high degree','Immune low degree')
xlabel('Time (weeks)','FontSize',24)
ax = gca;
ax.YAxis(1).Limits = ([0, 0.15]);
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
set(gca, 'FontSize', 13)
hold off


