% Code to reproduce figures 7 and 8 in Policy During an Epidemic With
% Super-Spreaders, Van Vlokhoven (2020)
% Evolution of epidemic with a social planner, with and without an ICU
% constraint

clear all

%% parameters
par.T = 100;    %(1 period is a week), so solve for T weeks
par.dt = 1/2;   %time steps
par.T_grid = 1:par.dt:par.T;
par.nT = length(par.T_grid);

par.beta = 0.96^(par.dt/52);      %dicounting

%%%%%%%
%set to 1 for no ICU limit
par.ICU_limit = 1;              %if more infected people than this limit death rate goes up
par.death_slope = 0.01*par.dt;  %increase in death rate once infected is above ICU limit 
%%%%%%%

par.phi = [0.07 0.2];   %weight utility function social good
par.nphi = length(par.phi);
par.g = [0.7 0.3];      %density phi
par.E_phi = par.phi*par.g'; %expectation phi
par.epsilon = 0.02;     %fraction infected initially

par.alpha = 0.7;        %returns to scale production function

%disease parameters
par.p = 0.07;   %transmission risk
par.lambda_R = 0.4975*par.dt;       %recovery rate
par.lambda_D = 0.0025*par.dt;       %death rate

%solve steady state (pre-disease onset) 
par.As = (par.E_phi/(1-par.E_phi))^(1-par.alpha);   %set A_s such that P_s is 1 in economy with no infections (Ar=L=1)
par.B_ss = ((par.As).^(par.alpha/(par.alpha-1))+par.As)./((1+(par.As).^(1/(par.alpha-1))).^par.alpha);  %Budget, Ps=1,L=1,Ar=1
par.cr_ss = (1-par.phi)*par.B_ss;   %consumption regular good
par.cs_ss = par.phi*par.B_ss;       %consumption social good
par.CS_ss = par.cs_ss*par.g';       %aggregate consumption social good

par.d_CS = 1;  %parameter that governs to what extent degree depends on aggregate consumption (in paper set to 1)

%%%%%%% value of life
par.value_life = 16*(52/par.dt)*par.B_ss;    %B_ss is income per capita per week (and 52 weeks in a year). 10 qaly (= 1million dollar) corresponds to 16 times GDP per capita


% set parameters to get following degree distribution in steady state
par.degree_target = [10 25]*par.dt;     %degree during steady state (before onset of epidemic)
par.a = (par.degree_target(2)-par.degree_target(1))/((par.CS_ss.^(par.d_CS))*par.B_ss*(par.phi(2)-par.phi(1)));
par.b = par.degree_target(1) - par.a*(par.CS_ss.^(par.d_CS))*par.cs_ss(1);
%degree = b + a * CS^d_CS * cs



%% solve social planner
%initial guess (comes from competitive eq)
%consumption social good
x0 = [0.0537934836845102 0.0509565970738223 0.0481240989933652 0.0455879509002401 0.0433806120924496 0.0414590122774540 0.0397997543042587 0.0386154558239445 0.0375872158742798 0.0368738773595352 0.0363096032062868 0.0360832084644850 0.0359249103704822 0.0359069820082782 0.0361437544830955 0.0363849291115811 0.0368237131677196 0.0372802198643945 0.0378440252977801 0.0385851330979718 0.0393100530431185 0.0400009010678835 0.0408511727758266 0.0416803790972235 0.0425635943769395 0.0434606613884978 0.0444162262585456 0.0454333869444404 0.0464189703111095 0.0473191068393498 0.0483797442941801 0.0492637754847680 0.0503002366768947 0.0511665196751093 0.0521660343234628 0.0530118529489164 0.0539339631321217 0.0548167698058303 0.0556814695946305 0.0564751694652340 0.0572621139257497 0.0580426993302775 0.0588170811581027 0.0594565580768280 0.0601300510970307 0.0608500823646801 0.0614497057981090 0.0620717448398177 0.0625379367347797 0.0631430062024752 0.0635834209397956 0.0641647056961158 0.0646096625187777 0.0650526607894515 0.0654422390395937 0.0658076610387708 0.0662473020120863 0.0665326928598424 0.0668914753522525 0.0671687159922288 0.0674592493262798 0.0677497584896850 0.0680264703260378 0.0683292104623642 0.0686038107853796 0.0687562256641731 0.0690299169936005 0.0691817362139894 0.0694702301083583 0.0696056380973052 0.0697561367272305 0.0698796890210077 0.0700301176635030 0.0701717666245924 0.0703221421105159 0.0704527532678113 0.0705928148928772 0.0707336075059520 0.0707392056600105 0.0708748980322548 0.0710069985905463 0.0711482330720886 0.0711682913662735 0.0712953215373470 0.0713177247832014 0.0714197884701075 0.0714450181912263 0.0715619528698315 0.0715617234654206 0.0715864377350525 0.0717088026773425 0.0717080896685308 0.0718553740837641 0.0718316120596415 0.0718311703888890 0.0718613776991321 0.0719749104299507 0.0719747918661790 0.0719744379373860 0.0720107394488026 0.0721216442848153 0.0721215628550763 0.0721213895482778 0.0721211549443186 0.0721200731255715 0.0722682882415981 0.0722681460490817 0.0722679343923741 0.0722677197519752 0.0722673244633764 0.0722662815146467 0.0722419954489464 0.0722409938091338 0.0722807682520521 0.0723872682892042 0.0723872732158218 0.0723872634349933 0.0723872235364765 0.0723872063222926 0.0723871493602401 0.0723871220140059 0.0723870414129273 0.0723869997636191 0.0723868866519154 0.0723866640724401 0.0723865751079167 0.0723863536869518 0.0723859142215996 0.0723852978909653 0.0723840663546642 0.0723819846307120 0.0724241285302921 0.0725339047073217 0.0725339063642119 0.0725339079227024 0.0725339093884126 0.0725339107666333 0.0725339120623472 0.0725339132802489 0.0725339144247624 0.0725339155000582 0.0725339165100687 0.0725339174585030 0.0725339183488599 0.0725339191844415 0.0725339199683642 0.0725339207035705 0.0725339213928391 0.0725339220387952 0.0725339226439188 0.0725339232105541 0.0725339237409171 0.0725339242371032 0.0725339247010941 0.0725339251347645 0.0725339255398880 0.0725339259181429 0.0725339262711175 0.0725339266003150 0.0725339269071580 0.0725339271929933 0.0725339274590952 0.0725339277066697 0.0725339279368582 0.0725339281507403 0.0725339283493373 0.0725339285336148 0.0725339287044858 0.0725339288628128 0.0725339290094109 0.0725339291450493 0.0725339292704542 0.0725339293863104 0.0725339294932635 0.0725339295919217 0.0725339296828577 0.0725339297666102 0.0725339298436858 0.0725339299145603 0.0725339299796804 0.0725339300394651 0.0725339300943071 0.0725339301445742 0.0725339301906104 0.0725339302327372 0.0725339302712550 0.0725339303064441 0.0725339303385655 0.0725339303678625 0.0725339303945615 0.0725339304188726 0.0725339304409913 0.0725339304610986 0.0725339304793624 0.0725339304959380 0.0725339305109692 0.0725339305245886 0.0725339305369189 0.0725339305480729;0.195438062984824 0.192608559171038 0.189283630170814 0.186661033587675 0.183610847165919 0.180470217636950 0.177817542310949 0.175875432212821 0.173800084542289 0.172476314073405 0.171648549657848 0.170976058482042 0.170732868529859 0.170972990443203 0.171576071524681 0.171494578153454 0.172566070552527 0.173473403169338 0.174642695575271 0.175894899136667 0.177046372701891 0.178499632523685 0.179850348427663 0.181306128784982 0.182710946762317 0.183997848975507 0.185214044994310 0.186627747274003 0.187983718114249 0.189276289300575 0.190311564226729 0.191375723990077 0.192311219356527 0.193590120181807 0.194315983872011 0.195101923019026 0.195837961724015 0.196973329742244 0.197511300304039 0.198156256286703 0.198787962053726 0.199408035645682 0.200017186631962 0.200272457296199 0.200889263834574 0.201102200122965 0.201749577372594 0.201975157524946 0.202401218188534 0.202588218102626 0.203169517217694 0.203432469254556 0.203679198462998 0.203923265693974 0.204157100888612 0.204057591560053 0.204300563268965 0.204602925888720 0.204764713024333 0.205048353644874 0.205055584739412 0.205209231826438 0.205495448974943 0.205371222107545 0.205653865282792 0.205595158287616 0.205877128586273 0.205817881150155 0.205859075413559 0.206039014408819 0.205977471974764 0.206324477286832 0.206263587696446 0.206331641533494 0.206263684061946 0.206486155315328 0.206426442560599 0.206472800184343 0.206444903060295 0.206709622779948 0.206654958548687 0.206680240196735 0.206670619391812 0.206613904236351 0.206601189659530 0.206880837836086 0.206871218347059 0.206887729359846 0.206887066144887 0.206861052809918 0.206821679507915 0.206819623056344 0.206755140461691 0.207108291315696 0.207109247288609 0.207142746977689 0.207095661331323 0.207095320184074 0.207094301813686 0.207076197582450 0.207030241542658 0.207030007792877 0.207029510303549 0.207028836856419 0.207025731419782 0.206964871023924 0.206964463807197 0.206963857656086 0.206963242960159 0.206962110916549 0.206959124077445 0.207335589390632 0.207333458988659 0.207344993039405 0.207304476986886 0.207304491095857 0.207304463085275 0.207304348822901 0.207304299524488 0.207304136395132 0.207304058080299 0.207303827252906 0.207303707976571 0.207303384044433 0.207302746615742 0.207302491836905 0.207301857725985 0.207300599174072 0.207298834110921 0.207295307206332 0.207289345512993 0.207261931432930 0.207239727735205 0.207239732469177 0.207239736922007 0.207239741109750 0.207239745047524 0.207239748749564 0.207239752229283 0.207239755499321 0.207239758571595 0.207239761457339 0.207239764167151 0.207239766711028 0.207239769098404 0.207239771338183 0.207239773438773 0.207239775408112 0.207239777253701 0.207239778982625 0.207239780601583 0.207239782116906 0.207239783534581 0.207239784860269 0.207239786099327 0.207239787256823 0.207239788337551 0.207239789346050 0.207239790286614 0.207239791163309 0.207239791979981 0.207239792740272 0.207239793447628 0.207239794105309 0.207239794716401 0.207239795283821 0.207239795810328 0.207239796298531 0.207239796750894 0.207239797169745 0.207239797557284 0.207239797915583 0.207239798246601 0.207239798552181 0.207239798834062 0.207239799093879 0.207239799333172 0.207239799553388 0.207239799755886 0.207239799941944 0.207239800112757 0.207239800269449 0.207239800413069 0.207239800544601 0.207239800664963 0.207239800775014 0.207239800875554 0.207239800967330 0.207239801051036 0.207239801127319 0.207239801196779 0.207239801259975 0.207239801317425 0.207239801369607 0.207239801416966 0.207239801459912 0.207239801498825 0.207239801534054 0.207239801565923]';
x0 = [x0(:,1); x0(:,2)];

% upper and lower bound on social good
lb = 0*ones(398,1);
ub = [0.4*ones(199,1); 0.5*ones(199,1)];

options = optimoptions(@fmincon,'MaxIterations',10000,'MaxFunctionEvaluations',1e9,'OptimalityTolerance',1e-9,'Display','iter');
[social_policy,social_val,exitflag3,output3] = fmincon(@(x) -Social_planner(x,par),x0,[],[],[],[],lb,ub,[],options);

c_s = [social_policy(1:199)  social_policy(200:end)];

%% simulate spread of disease
par.degree_exp = par.degree_target*par.g';

% evolution of states of disease over time for each type of agent
S_phi = NaN(par.nT,par.nphi);   %susceptible
S_phi(1,:) = (1-par.epsilon*par.degree_target/par.degree_exp); %reflects that those with a higher degree are more likely to be infected
R_phi = NaN(par.nT,par.nphi);   %recovered
R_phi(1,:) = 0;
I_phi = NaN(par.nT,par.nphi);   %infected
I_phi(1,:) = 1-S_phi(1,:);

frac_alive = NaN(par.nT,par.nphi);  
f = NaN(par.nT,1);      %probability that a given social interaction is with an infectious agent
C_S = NaN(par.nT,1);
degree = NaN(par.nT,2);
L = NaN(par.nT,1);

%% spread of disease
for t=1:par.nT-1 
    frac_alive(t,:) = S_phi(t,:) + I_phi(t,:) + R_phi(t,:);
    
    L(t) = frac_alive(t,:)*par.g';  %labor supply
    C_S(t) = (c_s(t,:).*frac_alive(t,:))*par.g';   %aggregate consumption
    degree(t,:) = par.b + par.a*(C_S(t).^(par.d_CS)).*c_s(t,:);
    
    f(t) = ((I_phi(t,:).*degree(t,:))*par.g')/((degree(t,:).*frac_alive(t,:))*par.g');  
    
    s_out = par.p*f(t)*degree(t,:).*S_phi(t,:);
    
    S_phi(t+1,:) = S_phi(t,:)-s_out;
    I_phi(t+1,:) = (1-(par.lambda_R+par.lambda_D))*I_phi(t,:)+s_out;
    R_phi(t+1,:) = R_phi(t,:)+par.lambda_R*I_phi(t,:);
end
frac_alive(par.nT,:) = S_phi(par.nT,:) + I_phi(par.nT,:) + R_phi(par.nT,:);

%% create plots
nT2=par.nT-100;
T_grid = par.T_grid;
% nT2=nT;

I_overall = I_phi*par.g';

figure(1)
yyaxis left
plot(T_grid(1:nT2),I_overall(1:nT2),'linewidth',2,'Color','b')
hold on
plot(T_grid(1:nT2),I_phi((1:nT2),2),'linewidth',2,'Color','r')
plot(T_grid(1:nT2),I_phi((1:nT2),1),'linewidth',2,'Color','g')
plot(T_grid(1:nT2),f(1:nT2),'linewidth',2,'Color','m')
ylabel('Fraction of Initial Population','FontSize',24)
yyaxis right
plot(T_grid(1:nT2),1-frac_alive(1:nT2,:)*par.g','linewidth',2,'Color','k')
ylabel('Deaths as Fraction of Initial Population','FontSize',24)
legend('Infected','Infected high degree','Infected low degree','Prob meeting is infected','Death (right axis)')
% legend('Infected','Death (right axis)')
xlabel('Time (weeks)','FontSize',24)
ax = gca;
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
ax.YAxis(1).Limits = ([0, 0.3]);
ax.YAxis(2).Limits = ([0, 0.005]);
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off


figure(2)
plot(T_grid(1:nT2),c_s(1:nT2,1)/par.cs_ss(1),'linewidth',2)
hold on
plot(T_grid(1:nT2),c_s(1:nT2,2)/par.cs_ss(2),'--','linewidth',2)
legend('Low degree','High degree')
ylabel('Cons social good relative to steady state','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
ax = gca;
ax.YAxis.Limits = ([0.4, 1.1]);
hold off


figure(3)
plot(T_grid(1:nT2),degree((1:nT2),1)/par.degree_target(1),'linewidth',2)
hold on
plot(T_grid(1:nT2),degree((1:nT2),2)/par.degree_target(2),'linewidth',2)
legend('Low degree','High degree')
ylabel('Degree relative to steady state','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
ax = gca;
ax.YAxis.Limits = ([0.4, 1.1]);
hold off

