% Code to reproduce figure 9 in Policy During an Epidemic With
% Super-Spreaders, Van Vlokhoven (2020)
% Evolution of epidemic when a planner has the option to set an upper bound
% to the consumption of the social good

clear all

%% parameters
par.T = 100;    %(1 period is a week), so solve for T weeks
par.dt = 1/2;   %time steps
par.T_grid = 1:par.dt:par.T;
par.nT = length(par.T_grid);

par.beta = 0.96^(par.dt/52);     %dicounting
  
par.phi = [0.07 0.2];   %weight utility function social good
par.nphi = length(par.phi);
par.g = [0.7 0.3];  %density phi
par.g = par.g/sum(par.g);
par.E_phi = par.phi*par.g'; %expectation phi

par.epsilon = 0.02;     %fraction infected initially

par.alpha = 0.7;         %returns to scale production function

%disease parameters
par.p = 0.07;       %transmission risk
par.lambda_R = 0.4975*par.dt;       %recovery rate
par.lambda_D = 0.0025*par.dt;       %death rate

%solve steady state (pre-disease onset) 
par.As = (par.E_phi/(1-par.E_phi))^(1-par.alpha);   %set A_s such that P_s is 1 in economy with no infections (Ar=L=1)
par.B_ss = ((par.As).^(par.alpha/(par.alpha-1))+par.As)./((1+(par.As).^(1/(par.alpha-1))).^par.alpha);  %Budget, Ps=1,L=1,Ar=1
par.cr_ss = (1-par.phi)*par.B_ss;   %consumption regular good
par.cs_ss = par.phi*par.B_ss;       %consumption social good
par.CS_ss = par.cs_ss*par.g';       %aggregate consumption social good

par.d_CS = 1;  %parameter that governs to what extent degree depends on aggregate consumption (in paper set to 1)

%%%%%%% value of life
par.value_life = 16*(52/par.dt)*par.B_ss;    %B_ss is income per capita per week (and 52 weeks in a year). 10 qaly (= 1million dollar) corresponds to 16 times GDP per capita


% set parameters to get following degree distribution in steady state
par.degree_target = [10 25]*par.dt;     %degree during steady state (before onset of epidemic)
par.a = (par.degree_target(2)-par.degree_target(1))/((par.CS_ss.^(par.d_CS))*par.B_ss*(par.phi(2)-par.phi(1)));
par.b = par.degree_target(1) - par.a*(par.CS_ss.^(par.d_CS))*par.cs_ss(1);
%degree = b + a * CS^d_CS * cs


%% Solve optimal policy
%number of of periods for which the policy cannot change
par.pol = [2 2 2 2 2 2 3 3 3 5 5 8 10 50 0.5]/par.dt;  

% initial guess on policy (i.e., max social consumption)
x0 = [0.19 0.18 0.17 0.16 0.15 0.14 0.14 0.14 0.15 0.16 0.17 0.18 0.19 0.20]; 

%lower bound and upper bound for maximum social consumption (for fmincon)
lb = 0.001*ones(length(par.pol)-1,1);
ub = 0.23*ones(length(par.pol)-1,1);

options = optimoptions(@fmincon,'MaxIterations',1000,'MaxFunctionEvaluations',600,'OptimalityTolerance',1e-6,'StepTolerance',1e-6,'UseParallel',true,'Display','iter');

[authoritarian_policy,authoritarian_val,exitflag,output] = fmincon(@(x) -Authoritarian(x,par),x0,[],[],[],[],lb,ub,[],options)


%% given the optimal policy, calculate consumption and evolution of epidemic
% parameters
T = par.T;    %(1 period is a week), so solve for T weeks
nT = par.nT;
dt = par.dt;
beta = par.beta;     %dicounting
phi = par.phi; 
nphi = par.nphi;
g = par.g;  %density phi
epsilon = par.epsilon;     %fraction infected initially
alpha = par.alpha;
As = par.As;   %set A_s such that P_s is 1 in economy with no infections (Ar=L=1)

%disease parameters
p = par.p;
lambda_R = par.lambda_R;
lambda_D = par.lambda_D;

%steady state values
B_ss = par.B_ss;  %Ps=1,L=1,Ar=1
cr_ss = par.cr_ss;
cs_ss = par.cs_ss;
CS_ss = par.CS_ss;

d_CS = par.d_CS;

a = par.a;
b = par.b;

degree_ss = b+a*cs_ss*(CS_ss.^(d_CS));
degree_ss_exp = degree_ss*g';

value_life = par.value_life;    %B_ss is income per capita per week. 10 qaly (= 1 million dollar) corresponds to 16 times GDP per capita
u_lowerbar = (1-beta)*value_life/B_ss - log(B_ss);

%states capturing evolution of epidemic
S_phi = NaN(nT,nphi);
S_phi(1,:) = (1-epsilon*degree_ss/degree_ss_exp); %reflects that those with a higher degree are more likely to be infected

I_phi = NaN(nT,nphi);
I_phi(1,:) = epsilon*degree_ss/degree_ss_exp;

cs_eq = NaN(nT,nphi);   %consumption of social good over time for each type

eps1 = 1;
iter1 = 0;
gamma = 1;    %update rate time variables
gammaP = 0.2;  

x = authoritarian_policy;
x = [x 0.24];   %last period no constraint
%construct the policy (for each period)
pol=par.pol;
kappa = NaN(1,nT);
t2 = 0;
for i_p = 1:length(pol)
    t1 = t2+1;
    t2 = t1 + pol(i_p) - 1;
    kappa(t1:t2) = x(i_p);    
end

% guess
f = [0.0252743187042286,0.0315361088717160,0.0384937589077554,0.0458856209819146,0.0534090551113552,0.0607311658411551,0.0675293822818887,0.0735622051250646,0.0786324875019579,0.0826044060288046,0.0854751533333284,0.0872706234549632,0.0880191731929458,0.0879085769457564,0.0869445209494787,0.0852170483177602,0.0829981245522543,0.0803002674826594,0.0772573395765691,0.0739547705282208,0.0705115108848695,0.0669984188091114,0.0634400834102358,0.0599122438679377,0.0564474824372110,0.0530775550217389,0.0498109970552529,0.0466837357732112,0.0436841243282816,0.0408660105360134,0.0382004992558925,0.0356710321427102,0.0333002703358012,0.0310701715387917,0.0289841225108286,0.0270424568152047,0.0252228201905419,0.0235285931395025,0.0219445214595160,0.0204739342894124,0.0191068191415239,0.0178358426051203,0.0166554334967115,0.0155504754773316,0.0145273679556938,0.0135745569816948,0.0126870142458037,0.0118610691644260,0.0110925541587225,0.0103776430299647,0.00971205543895595,0.00909000308337701,0.00850935055544727,0.00796991380799302,0.00746618183447577,0.00699481173980168,0.00655561513762544,0.00614576695458235,0.00576327549247168,0.00540539794425199,0.00507157660015885,0.00475811090801434,0.00446539849465172,0.00419133601957283,0.00393485767359254,0.00369476448084552,0.00346958442724948,0.00325961319904580,0.00306194723646885,0.00287649323340814,0.00270328286347493,0.00254034154805589,0.00238739402922272,0.00224413680466066,0.00210946142314619,0.00198306330450965,0.00186475668375714,0.00175356768132334,0.00164920056258254,0.00155075891846442,0.00145870662671513,0.00137213893978339,0.00129069727903355,0.00121420448951968,0.00114232678241126,0.00107455099118794,0.00101117599094138,0.000951463865230631,0.000895430332510578,0.000842609696589436,0.000792894216372060,0.000746169460045413,0.000702140070478877,0.000660715668886651,0.000621763597237614,0.000585229988415615,0.000550885683658638,0.000518607163978110,0.000488186478053196,0.000459528254417117,0.000432571963139843,0.000407240794393070,0.000383372265059624,0.000360884209750104,0.000339698882150445,0.000319748628173053,0.000300990387170909,0.000283365464502209,0.000266762046181693,0.000251122008616489,0.000236390484773483,0.000222515588547130,0.000209433281958183,0.000197178236859977,0.000185647623510999,0.000174828630753972,0.000164635342332353,0.000155031963586335,0.000145984698063585,0.000137461619666073,0.000129432620109271,0.000121869243807531,0.000114744675524297,0.000108033573701649,0.000101712053900457,9.57575610328782e-05,9.01488433547304e-05,8.48658639233226e-05,7.98897233638539e-05,7.52025489307624e-05,7.07875834639156e-05,6.66291459350626e-05,6.27140038634231e-05,5.90339683906835e-05,5.55733630349189e-05,5.23126535429221e-05,4.92400925624675e-05,4.63446648963853e-05,4.36160280070425e-05,4.10444624203175e-05,3.86208290039390e-05,3.63365312464825e-05,3.41834813377908e-05,3.21540692535698e-05,3.02411342927777e-05,2.84379386720775e-05,2.67381428842302e-05,2.51357825978315e-05,2.36252469261310e-05,2.22012579296538e-05,2.08588512450554e-05,1.95933577536085e-05,1.84003862186299e-05,1.72758068332094e-05,1.62157356286773e-05,1.52165197010547e-05,1.42747232177975e-05,1.33871141709190e-05,1.25506518453810e-05,1.17624749737459e-05,1.10198905496741e-05,1.03203632740756e-05,9.66150560868603e-06,9.04106841260570e-06,8.45693213797134e-06,7.90709856145241e-06,7.38968302870828e-06,6.90290718932425e-06,6.44509220007986e-06,6.01465237470508e-06,5.61008925856048e-06,5.22998610694913e-06,4.87300274604026e-06,4.53787079567045e-06,4.22338923359522e-06,3.92842028110259e-06,3.65188559027702e-06,3.39276271362342e-06,3.15008183723061e-06,2.92292275917411e-06,2.71041209542963e-06,2.51172069619112e-06,2.32606125615627e-06,2.15268610305597e-06,1.99088514945569e-06,1.83998399364046e-06,1.69934215620508e-06,1.56835143979880e-06,1.44643440031252e-06,1.33304291863920e-06,1.22765686297624e-06,1.12978283246652e-06,1.03895297378489e-06,9.54723863063914e-07,8.76675446310847e-07,8.04410032192724e-07,7.37551331754131e-07,6.75743540279571e-07,6.18650457116834e-07]';
L = [1,0.999975000000000,0.999944320772262,0.999907319764828,0.999863614662152,0.999813012628571,0.999755672054822,0.999691970476970,0.999622542897290,0.999548172797966,0.999469788309198,0.999388333387725,0.999304748097830,0.999219949816115,0.999134784628002,0.999050018775180,0.998966274588099,0.998884166340644,0.998804143255634,0.998726577941101,0.998651740152581,0.998579809159931,0.998510925792293,0.998445171169054,0.998382564121104,0.998323102431612,0.998266734093738,0.998213398862338,0.998163012846138,0.998115472303263,0.998070658836648,0.998028449849198,0.997988733676359,0.997951382798885,0.997916280342182,0.997883305199400,0.997852331269552,0.997823251804770,0.997795956583786,0.997770345209209,0.997746312566901,0.997723762260287,0.997702603061192,0.997682747062652,0.997664121804788,0.997646645983365,0.997630248418390,0.997614862478549,0.997600424827122,0.997586875340591,0.997574156834944,0.997562216284204,0.997551007495894,0.997540485946054,0.997530606820692,0.997521329984927,0.997512618602131,0.997504436281250,0.997496749896361,0.997489528273603,0.997482743294260,0.997476367033033,0.997470375308506,0.997464744063110,0.997459451095355,0.997454475279848,0.997449797959132,0.997445400621917,0.997441265005977,0.997437376208973,0.997433719694510,0.997430280293692,0.997427045396436,0.997424002806369,0.997421140829904,0.997418448750436,0.997415916287732,0.997413533601787,0.997411291713494,0.997409182049551,0.997407197324157,0.997405329664568,0.997403572005192,0.997401918109282,0.997400361680734,0.997398896746057,0.997397518212407,0.997396220553773,0.997394999234740,0.997393849482306,0.997392767273671,0.997391748714946,0.997390789907978,0.997389887470673,0.997389038125550,0.997388238592399,0.997387485935703,0.997386777393249,0.997386110248278,0.997385482140219,0.997384890834719,0.997384334167824,0.997383810005031,0.997383316488378,0.997382851860066,0.997382414458079,0.997382002711782,0.997381615099538,0.997381250129875,0.997380906501878,0.997380582986049,0.997380278420931,0.997379991709852,0.997379721816365,0.997379467722651,0.997379228507162,0.997379003218733,0.997378791054463,0.997378591256846,0.997378403111354,0.997378225944158,0.997378059119879,0.997377902039539,0.997377754138499,0.997377614884606,0.997377483776354,0.997377360341191,0.997377244133866,0.997377134734880,0.997377031749033,0.997376934804106,0.997376843549436,0.997376757654534,0.997376676807699,0.997376600706305,0.997376529060046,0.997376461612990,0.997376398123906,0.997376338365427,0.997376282123270,0.997376229195491,0.997376179391784,0.997376132532821,0.997376088449618,0.997376046982941,0.997376007982741,0.997375971307623,0.997375936824337,0.997375904407303,0.997375873938160,0.997375845305342,0.997375818403673,0.997375793133991,0.997375769402791,0.997375747121890,0.997375726208109,0.997375706582977,0.997375688172452,0.997375670906657,0.997375654719636,0.997375639549122,0.997375625336318,0.997375612025700,0.997375599564816,0.997375587904121,0.997375576996797,0.997375566798607,0.997375557267743,0.997375548364688,0.997375540052093,0.997375532294652,0.997375525058993,0.997375518313569,0.997375512028562,0.997375506175791,0.997375500728622,0.997375495661890,0.997375490951819,0.997375486575953,0.997375482513086,0.997375478743199,0.997375475247399,0.997375472007864,0.997375469007785,0.997375466231319,0.997375463663538,0.997375461290383,0.997375459098623,0.997375457075811,0.997375455210243,0.997375453490923,0.997375451907527,0.997375450450366,0.997375449110354,0.997375447878979,0.997375446748268,0.997375445710762,0.997375444759489,0.997375443887934]';
Ps = [0.927120075509146,0.914295372191969,0.900755699311531,0.888501671284327,0.876593804558082,0.866225995049189,0.856970409043824,0.849497960912450,0.843200448011832,0.838727548279146,0.835653425503534,0.833679724346914,0.832870113278464,0.832965471427779,0.834435728971348,0.835551448145170,0.838414521011413,0.841370907278822,0.845004357779187,0.849246414294413,0.853463554864283,0.857924250433844,0.862798615734999,0.867491328100412,0.872618980909487,0.877444199500986,0.882344135286599,0.887354441108444,0.892440965172141,0.897519624293433,0.902039958077445,0.906767826774937,0.911119517198258,0.915484312621682,0.920073320746903,0.923890530140902,0.927811274094640,0.931395692127294,0.935176380471816,0.938701769002761,0.942083122986862,0.945412835741664,0.947849662222716,0.951023565058839,0.953787540051018,0.956187958336386,0.958664969524082,0.961023938111015,0.963346554496083,0.965441031399850,0.967148509674378,0.968863379954282,0.970987969500087,0.972616064590501,0.973851664212448,0.975495616349396,0.976968220484238,0.978393086452068,0.979564783038367,0.980912645736359,0.981712909805110,0.982877479325009,0.983834784816814,0.984827717647646,0.985676966377497,0.986404835362600,0.987589373865170,0.987971425398085,0.988545453235288,0.989528594755279,0.989906104842987,0.990446363110315,0.991093838953819,0.991467969310683,0.991979500496486,0.992642086372318,0.992997521169611,0.993481855601402,0.993512961764826,0.994197620507823,0.994525569297555,0.994676293657113,0.994989059412448,0.995344558035112,0.995350344841757,0.996048683639804,0.996053988137502,0.996489952632512,0.996491081444096,0.996561668681710,0.996846266773636,0.996848340854391,0.996935602852332,0.997296817870909,0.997573491775964,0.997717283964915,0.997988341766137,0.997989236516748,0.997990133892184,0.998079384516811,0.998342796803836,0.998343444653965,0.998344057224336,0.998345531181048,0.998348546880219,0.998449367839441,0.998698186947822,0.998699428784955,0.998700593227249,0.998702690129341,0.998705687721111,0.998727263344119,0.999095286487123,0.999117923626692,0.999482062295431,0.999482065207351,0.999482148765149,0.999482401887546,0.999482404312016,0.999482870496529,0.999482872637773,0.999483524116402,0.999483776102764,0.999484437030338,0.999484795854115,0.999485713205941,0.999487496800640,0.999489993152043,0.999491020100772,0.999494935106302,0.999508076128903,0.999541701184656,0.999713892892540,0.999834843954959,0.999834844812869,0.999834845615409,0.999834846365354,0.999834847065448,0.999834847718375,0.999834848326738,0.999834848893049,0.999834849419723,0.999834849909070,0.999834850363299,0.999834850784514,0.999834851174717,0.999834851535812,0.999834851869607,0.999834852177817,0.999834852462070,0.999834852723907,0.999834852964787,0.999834853186093,0.999834853389132,0.999834853575141,0.999834853745289,0.999834853900678,0.999834854042351,0.999834854171292,0.999834854288427,0.999834854394628,0.999834854490717,0.999834854577468,0.999834854655606,0.999834854725812,0.999834854788725,0.999834854844945,0.999834854895030,0.999834854939505,0.999834854978856,0.999834855013539,0.999834855043978,0.999834855070565,0.999834855093666,0.999834855113619,0.999834855130736,0.999834855145307,0.999834855157596,0.999834855167847,0.999834855176287,0.999834855183120,0.999834855188533,0.999834855192699,0.999834855195772,0.999834855197894,0.999834855199194,0.999834855199785,0.999834855199773,0.999834855199251,0.999834855198301,0.999834855196999,0.999834855195409,0.999834855193591,0.999834855191595,0.999834855189467,0.999834855187245,0.999834855184963,0.999834855182651,0.999834855180334]';

%calculate Ps with constraint. cs_comp is cs_eq' from competitive 
cs_comp = [0.0538093815897652,0.0509678579621077,0.0481137149494357,0.0455801410895217,0.0433998904940908,0.0414581022282318,0.0399064713227310,0.0386166500008150,0.0375855370116467,0.0368711720717438,0.0363984391981394,0.0360780110387756,0.0360099527773174,0.0359027504929478,0.0361344846873983,0.0365643331077035,0.0368164264812568,0.0372766200263017,0.0378372071917200,0.0385783216377489,0.0393015625335663,0.0399948342183606,0.0408437934147908,0.0416796955935298,0.0425608257378750,0.0434545839408300,0.0444089372024227,0.0453372270899054,0.0464291492457145,0.0473136427361781,0.0482120093880770,0.0492565139754451,0.0501803367383745,0.0511556396447567,0.0521434334525691,0.0530067752997255,0.0539338142167822,0.0547362603677543,0.0556563926654739,0.0564819969164457,0.0572650471236147,0.0580419211445132,0.0586828432899614,0.0594523793302874,0.0600609141831246,0.0607471061302678,0.0613426213281847,0.0619403915544400,0.0625359220264452,0.0631403801128823,0.0635902596772427,0.0640364205883466,0.0644552540439626,0.0649009657689776,0.0653677902009424,0.0658075965115544,0.0661777972000116,0.0665314201641760,0.0668212098836829,0.0671689157243902,0.0674712298251632,0.0677501673769724,0.0680409906060564,0.0683287400815659,0.0684531897406805,0.0687559775346953,0.0690294858996024,0.0691814642458588,0.0693210097192468,0.0696052811313817,0.0697563084350363,0.0698967306554515,0.0700300306301291,0.0701802671975760,0.0703215062131755,0.0704466508025247,0.0705944614829611,0.0707339809554439,0.0707459767564601,0.0708748893798558,0.0710079269812523,0.0710187081589778,0.0711484551785704,0.0712957340325888,0.0712953974658806,0.0714206385282074,0.0714203287639181,0.0715622289646338,0.0715621829326595,0.0715904089060376,0.0717091813384172,0.0717090702177979,0.0717395178265656,0.0718499759957256,0.0718322222838717,0.0718617718144197,0.0719751530858501,0.0719751108873619,0.0719750676239696,0.0720110831607694,0.0721218954906440,0.0721218658411885,0.0721218377588891,0.0721217535336511,0.0721215693856403,0.0721601414673183,0.0722684951498661,0.0722684234701913,0.0722683562909114,0.0722682284995264,0.0722680421729171,0.0722666571009493,0.0722429351640732,0.0722414821497069,0.0723873293241172,0.0723873343446751,0.0723873338568963,0.0723873221392151,0.0723873263329812,0.0723873003407605,0.0723873040600689,0.0723872656442990,0.0723872528006291,0.0723872133630616,0.0723871932357481,0.0723871368824549,0.0723870244531778,0.0723868658645157,0.0723868019656790,0.0723865515291230,0.0723857055197454,0.0723835374635523,0.0724235733306179,0.0725338753648969,0.0725338769697741,0.0725338784810312,0.0725338799040726,0.0725338812439839,0.0725338825055522,0.0725338836932843,0.0725338848114236,0.0725338858639661,0.0725338868546749,0.0725338877870940,0.0725338886645614,0.0725338894902213,0.0725338902670356,0.0725338909977946,0.0725338916851271,0.0725338923315105,0.0725338929392790,0.0725338935106329,0.0725338940476465,0.0725338945522752,0.0725338950263630,0.0725338954716491,0.0725338958897742,0.0725338962822864,0.0725338966506465,0.0725338969962338,0.0725338973203503,0.0725338976242256,0.0725338979090213,0.0725338981758348,0.0725338984257028,0.0725338986596057,0.0725338988784698,0.0725338990831712,0.0725338992745380,0.0725338994533537,0.0725338996203591,0.0725338997762550,0.0725338999217041,0.0725339000573335,0.0725339001837361,0.0725339003014728,0.0725339004110740,0.0725339005130413,0.0725339006078487,0.0725339006959445,0.0725339007777521,0.0725339008536715,0.0725339009240806,0.0725339009893359,0.0725339010497741,0.0725339011057125,0.0725339011574504,0.0725339012052697,0.0725339012494361,0.0725339012901994,0.0725339013277947,0.0725339013624431,0.0725339013943521,0.0725339014237168,0.0725339014507198,0.0725339014755327,0.0725339014983158,0.0725339015192196,0.0725339015383845;0.195174768905733,0.192144302228935,0.189279346709499,0.186245697629033,0.183137603410927,0.180447448066789,0.177739685661639,0.175566120424029,0.173745975312523,0.172413354824702,0.171518159474082,0.171108709143459,0.170618225774302,0.170870995167944,0.171442063823749,0.171359147295517,0.172449383267551,0.173359994438879,0.174530957241337,0.175756470801142,0.176902356928313,0.178340095258774,0.179744345559969,0.181122216783547,0.182585456603108,0.183877693413914,0.185104675856983,0.186545922501337,0.187575200298385,0.189193764376690,0.190302305920594,0.191291933143487,0.192282400873407,0.193204210442455,0.194263025099228,0.195053634311417,0.195783055262974,0.196539519354924,0.197223552129205,0.197924901518563,0.198624684785541,0.199307507584223,0.199636131418287,0.200237536230942,0.200900810900541,0.201114895188362,0.201581258311342,0.202001783103547,0.202381825165622,0.202568034476336,0.202821344968715,0.203067461270837,0.203727352187354,0.203847557227834,0.203804804394160,0.204044949596310,0.204313249425414,0.204586695341938,0.204782421844551,0.205042855990477,0.204912654900576,0.205204362343620,0.205238584327510,0.205363685776983,0.205710072857152,0.205588337009535,0.205869769406065,0.205810999270287,0.205928126446359,0.206031911723488,0.205971930986739,0.206062930701869,0.206257277445112,0.206196213811933,0.206265778442346,0.206480408230780,0.206421426273831,0.206473890290801,0.206468256056979,0.206703533924375,0.206648165120847,0.206738788303552,0.206680885399280,0.206615099647923,0.206614124277308,0.206874144974658,0.206876950135700,0.206888527558113,0.206888394478309,0.206875392766064,0.206822771637703,0.206822451144929,0.206806362575911,0.206739608116827,0.207103560753494,0.207146481735021,0.207096359532733,0.207096238113703,0.207096113630591,0.207079640458386,0.207030962646619,0.207030877535776,0.207030796923565,0.207030555149108,0.207030026539319,0.207011416155507,0.206965463576260,0.206965258297084,0.206965065906323,0.206964699932143,0.206964166322401,0.206960199696815,0.207330780032204,0.207327326139076,0.207304651780203,0.207304666158203,0.207304664761289,0.207304631203900,0.207304643214112,0.207304568776938,0.207304579428387,0.207304469412340,0.207304432630316,0.207304319688019,0.207304262046914,0.207304100660937,0.207303778683150,0.207303324512956,0.207303141517863,0.207302424311364,0.207300001488460,0.207293792554682,0.207261980611978,0.207239643899706,0.207239648485069,0.207239652802946,0.207239656868779,0.207239660697097,0.207239664301578,0.207239667695098,0.207239670889782,0.207239673897046,0.207239676727642,0.207239679391697,0.207239681898747,0.207239684257775,0.207239686477245,0.207239688565127,0.207239690528935,0.207239692375744,0.207239694112226,0.207239695744666,0.207239697278990,0.207239698720786,0.207239700075323,0.207239701347569,0.207239702542212,0.207239703663675,0.207239704716133,0.207239705703525,0.207239706629572,0.207239707497788,0.207239708311490,0.207239709073814,0.207239709787722,0.207239710456016,0.207239711081342,0.207239711666203,0.207239712212966,0.207239712723868,0.207239713201026,0.207239713646443,0.207239714062012,0.207239714449524,0.207239714810675,0.207239715147065,0.207239715460212,0.207239715751547,0.207239716022425,0.207239716274127,0.207239716507863,0.207239716724776,0.207239716925945,0.207239717112388,0.207239717285069,0.207239717444893,0.207239717592715,0.207239717729342,0.207239717855532,0.207239717971998,0.207239718079413,0.207239718178409,0.207239718269578,0.207239718353476,0.207239718430628,0.207239718501522,0.207239718566617,0.207239718626342,0.207239718681099]';
cs_demand_tmp = (min(cs_comp,repmat(kappa',1,2))*g').*L;
Ps_tmp = ((max((As^(1/alpha)) * L./(cs_demand_tmp.^(1/alpha)) - 1,0.00001)).^(alpha-1))/As;    %Price at which social market clears. make sure that term to the power is positive
Ps = min(Ps,Ps_tmp);  
CS = cs_demand_tmp;

% construct the grid
dS = 0.025;
S_grid = 0:dS:1;     

dI = 0.03; 
I_grid = 0:dI:0.3; 

T_grid = 1:dt:T;

nS = length(S_grid);    %make sure nS  = nR
nI = length(I_grid);
nT = length(T_grid);

xs_grid = 0.01:(1.04/440):1.05;   %share of phi spent on social good

[I_mat,S_mat,XS_mat] = meshgrid(I_grid,S_grid,xs_grid);
I_tmp = I_mat(:,:,1);
S_tmp = S_mat(:,:,1);

alive = min(1-(1-S_mat - I_mat)*lambda_D/(lambda_D+lambda_R),1); 
Itilde_mat = I_mat./alive;

L1 = L;
Ps1 = Ps;
f1 = f;
CS1 = CS;

epsL = 1;
epsf = 1;
epsP = 1;
epsCS = 1;
a0 = a;  
while (epsL > 1e-3 || epsf > 1e-4 || epsP > 1e-3 || epsCS > 1e-3) && iter1 < 100      
    iter1 = iter1 + 1;

    L = gamma*L1 + (1-gamma)*L;
    Ps = gammaP*Ps1 + (1-gammaP)*Ps;
    f = gamma*f1 + (1-gamma)*f;
    CS = gammaP*CS1 + (1-gammaP)*CS;
    a = a0*CS.^d_CS;
    
    %income per capita per period
    B = (((As*Ps).^(alpha/(alpha-1))+As*Ps)./((1+(As*Ps).^(1/(alpha-1))).^alpha)).*(L.^(alpha-1));   
     
    V = NaN(nS,nI,nT,nphi);
    cs_fin = NaN(nS,nI,nT,nphi);
    cs_fin(:,:,nT,1)= repmat(phi(1)*B(end)/Ps(end),nS,nI,1);
    cs_fin(:,:,nT,2)= repmat(phi(2)*B(end)/Ps(end),nS,nI,1);
    % assume that after last period everyone that is I moves to R
    % in last period, consume steady state
    V(:,:,nT,1) = ( (1-phi(1))*log((B(end)-Ps(end)*cs_fin(:,:,nT,1))/(1-phi(1))) + phi(1)*log(cs_fin(:,:,nT,1)/phi(1)) + u_lowerbar)/(1-beta); 
    V(:,:,nT,2) = ( (1-phi(2))*log((B(end)-Ps(end)*cs_fin(:,:,nT,2))/(1-phi(2))) + phi(2)*log(cs_fin(:,:,nT,2)/phi(2)) + u_lowerbar)/(1-beta); 

    for i_phi = 1:nphi   
        for t = nT-1:-1:1       
            cs = XS_mat*phi(i_phi)*B(t)/Ps(t); %consumption social good    
            cr = (1-XS_mat*phi(i_phi))*B(t);    %B(t)-cs*Ps(t); %
            degree = b+a(t)*cs;
            S_out = p*f(t)*degree.*S_mat;
            Sprime = max(S_mat-S_out,S_grid(1));
            Iprime = I_mat*(1-lambda_R-lambda_D)+S_out;

            U = -1e10*(cs>kappa(t)) + phi(i_phi)*log(cs) + (1-phi(i_phi))*log(cr) - phi(i_phi)*log(phi(i_phi)) - (1-phi(i_phi))*log(1-phi(i_phi)) + u_lowerbar;
            Wprime = interp2(I_tmp,S_tmp,V(:,:,t+1,i_phi),Iprime,Sprime,'linear',-inf);

            %optimize over consumption
            Wmatrix = U + beta*(1-lambda_D*Itilde_mat).*Wprime;     %
            [Wtemp,cs_ind] = max(Wmatrix,[],3);
            V(:,:,t,i_phi) = Wtemp;
            cs_fin(:,:,t,i_phi) = xs_grid(cs_ind)*phi(i_phi)*B(t)/Ps(t);
        end

         %solve for time path S and I
        for t=1:nT-1
            cs_eq(t,i_phi) = interp2(I_tmp,S_tmp,cs_fin(:,:,t,i_phi),I_phi(t,i_phi),S_phi(t,i_phi),'linear');

            S_out = p*f(t)*(b+a(t)*cs_eq(t,i_phi)).*S_phi(t,i_phi);
            S_phi(t+1,i_phi) = S_phi(t,i_phi) - S_out; 
            I_phi(t+1,i_phi) = I_phi(t,i_phi)*(1-lambda_R-lambda_D)+S_out;
        end
        cs_eq(nT,i_phi) = interp2(I_tmp,S_tmp,cs_fin(:,:,nT,i_phi),I_phi(nT,i_phi),S_phi(nT,i_phi),'linear');

    end
    degree_eq = b+repmat(a,1,nphi).*cs_eq;

    frac_alive = 1-(1-S_phi-I_phi)*lambda_D/(lambda_R+lambda_D);

    L1 = frac_alive*g';         %mass of agents alive (and labor supply)
    f1 = ((degree_eq.*I_phi)*g')./((degree_eq.*frac_alive)*g'); %probability that a person you meet is infected 

    cs_demand = (cs_eq.*frac_alive)*g';
    Ps1 = ((max((As^(1/alpha)) * L1./(cs_demand.^(1/alpha)) - 1,0.00001)).^(alpha-1))/As;    %Price at which social market clears. make sure that term to the power is positive

    CS1 = cs_demand;

    eps1 = norm(L1-L) + norm(f1-f) + norm(Ps1-Ps);
    epsL = norm(L1-L);
    epsf = norm(f1-f);
    epsP = norm(Ps1-Ps);
    epsCS = norm(CS1-CS);
    fprintf('iter big loop=%d epsL=%1.8f epsf=%1.8f epsP=%1.8f epsCS=%1.8f\n',iter1,epsL,epsf,epsP,epsCS) 
end

B_fin = B;

I_overall = I_phi*g';

value1 = interp2(I_tmp,S_tmp,V(:,:,1,1),I_phi(1,1),S_phi(1,1),'linear');
value2 = interp2(I_tmp,S_tmp,V(:,:,1,2),I_phi(1,2),S_phi(1,2),'linear');
        
value_auhthoritarian = g(1) * value1 +  g(2) * value2;

%% create plots
nT2 = nT-100;

figure(1)
yyaxis left
plot(T_grid(1:nT2),I_overall(1:nT2),'linewidth',2,'Color','b')
hold on
plot(T_grid(1:nT2),I_phi((1:nT2),2),'linewidth',2,'Color','r')
plot(T_grid(1:nT2),I_phi((1:nT2),1),'linewidth',2,'Color','g')
plot(T_grid(1:nT2),f(1:nT2),'linewidth',2,'Color','m')
ylabel('Fraction of Initial Population','FontSize',24)
yyaxis right
plot(T_grid(1:nT2),1-L(1:nT2),'linewidth',2,'Color','k')
ylabel('Deaths as Fraction of Initial Population','FontSize',24)
legend('Infected','Infected high degree','Infected low degree','Prob meeting is infected','Death (right axis)')
% legend('Infected','Death (right axis)')
xlabel('Time (weeks)','FontSize',24)
ax = gca;
ax.YAxis(1).Color = 'k';
ax.YAxis(2).Color = 'k';
ax.YAxis(1).Limits = ([0, 0.3]);
ax.YAxis(2).Limits = ([0, 0.005]);
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off

% kappa(kappa>0.193) = cs_ss(2);
% kappa(1:16) = cs_ss(2);
figure(2)
plot(T_grid(1:nT2),kappa(1:nT2),'-','linewidth',3,'Color','k')
hold on
plot(T_grid(1:nT2),cs_eq((1:nT2),1),'linewidth',2,'Color',[0 0.4470 0.7410])
plot(T_grid(1:nT2),repmat(cs_ss(1),nT2,1),'--','linewidth',2,'Color',[0 0.4470 0.7410],'HandleVisibility','off')
plot(T_grid(1:nT2),cs_eq((1:nT2),2),'linewidth',2,'Color',[0.8500 0.3250 0.0980])
plot(T_grid(1:nT2),repmat(cs_ss(2),nT2,1),'--','linewidth',2,'Color',[0.8500 0.3250 0.0980],'HandleVisibility','off')
legend('Constraint','Consumption (low degree)','Consumption (high degree)')
ylabel('Cons social good','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
xt = get(gca, 'XTick');
ax = gca;
ax.YAxis.Limits = ([0, 0.29]);
set(gca, 'FontSize', 13)
hold off

figure(3)
plot(T_grid(1:nT2),degree_eq((1:nT2),1)/degree_ss(1),'linewidth',2)
hold on
plot(T_grid(1:nT2),degree_eq((1:nT2),2)/degree_ss(2),'linewidth',2)
legend('Low degree','High degree')
ylabel('Degree relative to steady state','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off

figure(4)
plot(T_grid(1:nT2),B_fin(1:nT2)/B_ss,'linewidth',2,'Color','k')
hold on
ylabel('Income per capita relative to steady state','FontSize',24)
xlabel('Time (weeks)','FontSize',24)
xt = get(gca, 'XTick');
set(gca, 'FontSize', 13)
hold off


